<?php
/**
 * Plugin Name: JulyPay WooCommerce Gateway
 * Description: JulyPay payment gateway for WooCommerce. Collects mobile money payment. Supports MTN and Airtel Uganda mobile money.
 * Version: 1.0.0
 * Author: JulyBrands Digital
 */

if (!defined('ABSPATH')) exit;

add_action('plugins_loaded', 'julypay_init_gateway', 11);
function julypay_init_gateway() {
    if (!class_exists('WC_Payment_Gateway')) return;

    class WC_Gateway_JulyPay extends WC_Payment_Gateway {
        public function __construct() {
            $this->id = 'julypay';
            $this->has_fields = true;
            $this->method_title = 'JulyPay';
            $this->method_description = 'Accept mobile money payments via JulyPay STK Push.';

            // Settings
            $this->init_form_fields();
            $this->init_settings();

            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');
            $this->api_key = $this->get_option('api_key');
            $this->test_mode = 'yes' === $this->get_option('test_mode');
            $this->base_url = $this->test_mode ? 'https://app.julypay.net/api/v1' : 'https://app.julypay.net/api/v1'; // same base for now

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            add_action('wp_enqueue_scripts', array($this, 'frontend_scripts'));
            add_action('woocommerce_thankyou_' . $this->id, array($this, 'thankyou_page'));

            // Webhook endpoint
            add_action('rest_api_init', function() {
                register_rest_route('julypay/v1', '/webhook', array(
                    'methods' => 'POST',
                    'callback' => array($this, 'handle_webhook'),
                ));
            });
        }

        public function init_form_fields() {
            $this->form_fields = array(
                'enabled' => array(
                    'title' => 'Enable/Disable',
                    'type' => 'checkbox',
                    'label' => 'Enable JulyPay Payment',
                    'default' => 'yes'
                ),
                'title' => array(
                    'title' => 'Title',
                    'type' => 'text',
                    'description' => 'Shown to the customer during checkout',
                    'default' => 'Pay with JulyPay (STK Push)'
                ),
                'description' => array(
                    'title' => 'Description',
                    'type' => 'textarea',
                    'default' => 'You will be prompted on your phone to approve the payment.'
                ),
                'api_key' => array(
                    'title' => 'API Key (Bearer)',
                    'type' => 'text',
                    'description' => 'Your JulyPay API key (use Dashboard to create an API key)'
                ),
                'test_mode' => array(
                    'title' => 'Test Mode',
                    'type' => 'checkbox',
                    'label' => 'Enable Test Mode',
                    'default' => 'yes'
                )
            );
        }

        public function frontend_scripts() {
            if (is_checkout()) {
                wp_register_script('julypay-frontend', plugins_url('julypay-frontend.js', __FILE__), array('jquery'), '1.0', true);
                wp_localize_script('julypay-frontend', 'JulyPayData', array(
                    'ajax_url' => admin_url('admin-ajax.php'),
                ));
                wp_enqueue_script('julypay-frontend');
                wp_register_style('julypay-style', plugins_url('julypay-style.css', __FILE__));
                wp_enqueue_style('julypay-style');
            }
        }

        public function payment_fields() {
            // show the phone input field
            echo '<p>' . esc_html($this->description) . '</p>';
            woocommerce_form_field('julypay_phone', array(
                'type' => 'text',
                'label' => __('Phone number', 'julypay'),
                'placeholder' => 'e.g. 0701234567',
                'required' => true,
            ), '');
        }

        // Normalize phone to start with 07
        private function normalize_phone($phone) {
            $p = preg_replace('/[^0-9_+]/', '', $phone); // keep digits, underscores, plus
            // handle leading +
            $p = preg_replace('/^\+/', '', $p);
            // handle _256 or 256 prefix
            if (strpos($p, '_256') === 0) {
                $p = substr($p, 4);
            }
            if (strpos($p, '256') === 0) {
                $p = '0' . substr($p, 3);
            }
            // if starts with 0 already, keep
            if (strpos($p, '0') === 0) {
                return $p;
            }
            // fallback: if length is 9 (e.g. 701234567) prefix 0
            if (strlen($p) === 9) return '0' . $p;
            return $p; // last resort
        }

        public function validate_fields() {
            if (empty($_POST['julypay_phone'])) {
                wc_add_notice('Please enter phone number.', 'error');
                return false;
            }
            return true;
        }

        public function process_payment($order_id) {
            $order = wc_get_order($order_id);

            $phone_raw = isset($_POST['julypay_phone']) ? wc_clean(wp_unslash($_POST['julypay_phone'])) : '';
            $phone = $this->normalize_phone($phone_raw);

            // amount (JulyPay expects integer UGX)
            $amount = round($order->get_total());

            $body = array(
                'customer_phone' => $this->convert_phone_to_international($phone), 
                'amount' => intval($amount),
                'description' => 'Order #' . $order->get_order_number(),
                'customer_name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name()
            );

            $response = wp_remote_post($this->base_url . '/wallet/collect-payment', array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $this->api_key,
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json'
                ),
                'body' => wp_json_encode($body),
                'timeout' => 45,
            ));

            if (is_wp_error($response)) {
                wc_add_notice('Payment error: ' . $response->get_error_message(), 'error');
                return;
            }

            $code = wp_remote_retrieve_response_code($response);
            $data = json_decode(wp_remote_retrieve_body($response), true);

            if ($code === 200 && !empty($data['success'])) {
                $tx = $data['data'];
                // store transaction id on order for polling
                $order->update_meta_data('_julypay_transaction_id', $tx['transaction_id']);
                $order->update_meta_data('_julypay_reference', $tx['reference']);
                $order->save();

                // reduce stock, set on-hold
                wc_reduce_stock_levels($order_id);
                $order->update_status('on-hold', 'Awaiting JulyPay payment. Transaction: ' . $tx['transaction_id']);

                // return redirect to thankyou where JS will poll for result and show countdown
                return array(
                    'result' => 'success',
                    'redirect' => $this->get_return_url($order)
                );
            } else {
                $message = isset($data['message']) ? $data['message'] : 'Payment initiation failed';
                wc_add_notice('JulyPay error: ' . $message, 'error');
                return;
            }
        }

        private function convert_phone_to_international($phone) {
            // If phone starts with 0, convert to 256 format
            $p = preg_replace('/[^0-9]/', '', $phone);
            if (strpos($p, '0') === 0) {
                return '256' . substr($p, 1);
            }
            if (strpos($p, '256') === 0) return $p;
            // fallback
            return $p;
        }

        public function thankyou_page($order_id) {
            $order = wc_get_order($order_id);
            $tx = $order->get_meta('_julypay_transaction_id');
            if (!$tx) return;

            // output container for JS-driven countdown and status
            echo '<div id="julypay-status" data-transaction="' . esc_attr($tx) . '">';
            echo '<p>We have sent a payment prompt to your phone. Please approve the payment. Time left: <span id="julypay-countdown">60</span>s</p>';
            echo '<div id="julypay-result"></div>';
            echo '</div>';

            // inline script to poll status every 5s and update order when completed
            ?>
            <script>
            (function(){
                var tx = document.getElementById('julypay-status').getAttribute('data-transaction');
                var countdownEl = document.getElementById('julypay-countdown');
                var resultEl = document.getElementById('julypay-result');
                var timeLeft = 60; // seconds to wait before showing manual action

                function updateCountdown(){
                    if (timeLeft <= 0) return;
                    timeLeft--;
                    countdownEl.textContent = timeLeft;
                }
                var timer = setInterval(updateCountdown, 1000);

                var poll = setInterval(function(){
                    fetch('<?php echo esc_url($this->base_url); ?>' + '/wallet/collections/' + tx + '/status', {
                        headers: {
                            'Authorization': 'Bearer <?php echo esc_js($this->api_key); ?>',
                            'Accept': 'application/json'
                        }
                    }).then(function(r){ return r.json() }).then(function(data){
                        if (!data) return;
                        if (data.status === 'completed'){
                            clearInterval(poll);
                            clearInterval(timer);
                            countdownEl.textContent = '0';
                            resultEl.innerHTML = '<p>Payment received. Finalizing order...</p>';
                            // call ajax to mark order complete
                            fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>?action=julypay_mark_complete&order_id=<?php echo esc_js($order_id); ?>')
                                .then(function(){ window.location.reload(); });
                        } else if (data.status === 'failed'){
                            clearInterval(poll);
                            clearInterval(timer);
                            resultEl.innerHTML = '<p>Payment failed: ' + (data.failure_reason || 'Unknown') + '</p>';
                        } else {
                            // still processing
                        }
                    }).catch(function(e){
                        console.log('Poll error', e);
                    });
                }, 5000);
            })();
            </script>
            <?php
        }

        public function handle_webhook($request) {
            // Basic webhook handler - JulyPay should send events. Verify with your webhook secret if available.
            $body = $request->get_json_params();
            if (empty($body)) return new WP_REST_Response(array('ok' => false), 400);

            // Example: when a collection completes, find order by transaction id and mark complete
            if (!empty($body['transaction_id']) && !empty($body['status'])){
                $tx = $body['transaction_id'];
                $orders = wc_get_orders(array(
                    'limit' => 1,
                    'meta_key' => '_julypay_transaction_id',
                    'meta_value' => $tx
                ));
                if (!empty($orders)){
                    $order = $orders[0];
                    if ($body['status'] === 'completed'){
                        $order->payment_complete();
                        $order->add_order_note('JulyPay webhook: payment completed.');
                    } elseif ($body['status'] === 'failed'){
                        $order->update_status('failed', 'JulyPay webhook: payment failed.');
                    }
                }
            }

            return new WP_REST_Response(array('ok' => true), 200);
        }
    }

    // add gateway
    function julypay_add_gateway($methods) {
        $methods[] = 'WC_Gateway_JulyPay';
        return $methods;
    }
    add_filter('woocommerce_payment_gateways', 'julypay_add_gateway');

    // AJAX handler to mark order complete after polling success
    add_action('wp_ajax_julypay_mark_complete', 'julypay_mark_complete');
    add_action('wp_ajax_nopriv_julypay_mark_complete', 'julypay_mark_complete');
    function julypay_mark_complete(){
        if (empty($_GET['order_id'])) wp_die('Missing order id');
        $order_id = intval($_GET['order_id']);
        $order = wc_get_order($order_id);
        if (!$order) wp_die('Order not found');
        $order->payment_complete();
        $order->add_order_note('Marked complete via JulyPay polling.');
        echo 'ok';
        wp_die();
    }
}

?>
